import React from 'react'
import { Link, useNavigate } from 'react-router-dom'
import { useLanguage } from '../contexts/LanguageContext'
import './News.css'

const News = () => {
  const { t } = useLanguage()
  const navigate = useNavigate()

  const handleReadMoreClick = (link) => {
    navigate(link)
    window.scrollTo({ top: 0, behavior: 'smooth' })
  }

  
  const newsArticles = [
    {
      id: 1,
      title: t('news.articles.privatePension.title'),
      excerpt: t('news.articles.privatePension.excerpt'),
      content: 'Full article content would go here...',
      image: '/assets/category-5.png',
      date: '2024-01-15',
      category: t('news.categories.privatePension'),
      readTime: t('news.articles.privatePension.readTime'),
      author: t('news.articles.privatePension.author')
    },
    {
      id: 2,
      title: t('news.articles.investment.title'),
      excerpt: t('news.articles.investment.excerpt'),
      content: 'Full article content would go here...',
      image: '/assets/category-3.png',
      date: '2024-01-10',
      category: t('news.categories.investment'),
      readTime: t('news.articles.investment.readTime'),
      author: t('news.articles.investment.author')
    },
    {
      id: 3,
      title: t('news.articles.banking.title'),
      excerpt: t('news.articles.banking.excerpt'),
      content: 'Full article content would go here...',
      image: '/assets/category-4.png',
      date: '2024-01-05',
      category: t('news.categories.banking'),
      readTime: t('news.articles.banking.readTime'),
      author: t('news.articles.banking.author')
    },
    {
      id: 4,
      title: t('news.articles.lifeInsurance.title'),
      excerpt: t('news.articles.lifeInsurance.excerpt'),
      content: 'Full article content would go here...',
      image: '/assets/category-2.png',
      date: '2023-12-28',
      category: t('news.categories.lifeInsurance'),
      readTime: t('news.articles.lifeInsurance.readTime'),
      author: t('news.articles.lifeInsurance.author')
    },
    {
      id: 5,
      title: t('news.articles.insurance.title'),
      excerpt: t('news.articles.insurance.excerpt'),
      content: 'Full article content would go here...',
      image: '/assets/category-1.png',
      date: '2023-12-20',
      category: t('news.categories.insurance'),
      readTime: t('news.articles.insurance.readTime'),
      author: t('news.articles.insurance.author')
    }
  ]

  const categories = [
    { key: 'All', label: t('news.categories.all') },
    { key: t('news.categories.privatePension'), label: t('news.categories.privatePension') },
    { key: t('news.categories.investment'), label: t('news.categories.investment') },
    { key: t('news.categories.banking'), label: t('news.categories.banking') },
    { key: t('news.categories.lifeInsurance'), label: t('news.categories.lifeInsurance') },
    { key: t('news.categories.insurance'), label: t('news.categories.insurance') }
  ]
  const [selectedCategory, setSelectedCategory] = React.useState('All')

  const filteredArticles = selectedCategory === 'All' 
    ? newsArticles 
    : newsArticles.filter(article => article.category === selectedCategory)

  const formatDate = (dateString) => {
    const options = { year: 'numeric', month: 'long', day: 'numeric' }
    return new Date(dateString).toLocaleDateString(undefined, options)
  }

  return (
    <div className="news-page">
      {/* Page Header */}
      <section className="page-header">
        <div className="container">
          <div className="page-header-content">
            <h1 className="page-title">{t('newsPage.title')}</h1>
            <p className="page-subtitle">
              {t('newsPage.subtitle')}
            </p>
          </div>
        </div>
      </section>

      {/* Featured Article */}
      {newsArticles.length > 0 && (
        <section className="section">
          <div className="container">
            <div className="featured-article">
              <div className="featured-content">
                <div className="featured-meta">
                  <span className="featured-category">{newsArticles[0].category}</span>
                  <span className="featured-date">{formatDate(newsArticles[0].date)}</span>
                </div>
                <h2 className="featured-title">{newsArticles[0].title}</h2>
                <p className="featured-excerpt">{newsArticles[0].excerpt}</p>
                <div className="featured-author-info">
                  <span className="read-time">{newsArticles[0].readTime}</span>
                </div>
                <button onClick={() => handleReadMoreClick(`/news/1`)} className="btn btn-primary" style={{ border: 'none', cursor: 'pointer' }}>
                  {t('news.readMore')}
                </button>
              </div>
              <div className="featured-image">
                <img src={newsArticles[0].image} alt={newsArticles[0].title} />
              </div>
            </div>
          </div>
        </section>
      )}

      {/* Category Filter */}
      <section className="section bg-light">
        <div className="container">
          <div className="category-filter">
            <h3 className="filter-title">{t('newsPage.filterByCategory')}</h3>
            <div className="category-buttons">
              {categories.map(category => (
                <button
                  key={category.key}
                  className={`category-btn ${selectedCategory === category.key ? 'active' : ''}`}
                  onClick={() => setSelectedCategory(category.key)}
                >
                  {category.label}
                </button>
              ))}
            </div>
          </div>

          {/* Articles Grid */}
          <div className="news-grid">
            {filteredArticles.map((article) => (
              <article key={article.id} className="news-card">
                <div className="news-image">
                  <img src={article.image} alt={article.title} />
                  <div className="news-category">{article.category}</div>
                </div>
                
                <div className="news-content">
                  <div className="news-meta">
                    <span className="news-date">{formatDate(article.date)}</span>
                    <span className="news-read-time">{article.readTime}</span>
                  </div>
                  
                  <h3 className="news-title">
                    <button onClick={() => handleReadMoreClick(`/news/${article.id}`)} style={{ background: 'none', border: 'none', cursor: 'pointer', padding: 0, textAlign: 'left', color: 'inherit', fontSize: 'inherit', fontWeight: 'inherit' }}>
                      {article.title}
                    </button>
                  </h3>
                  
                  <p className="news-excerpt">{article.excerpt}</p>
                  
                  <div className="news-footer">
                    <button onClick={() => handleReadMoreClick(`/news/${article.id}`)} className="news-read-more" style={{ background: 'none', border: 'none', cursor: 'pointer', display: 'flex', alignItems: 'center', gap: '0.5rem' }}>
                      {t('news.readMore')}
                      <img src="/assets/arrow.svg" alt="" />
                    </button>
                  </div>
                </div>
              </article>
            ))}
          </div>

          {filteredArticles.length === 0 && (
            <div className="no-articles">
              <p>{t('news.noArticlesInCategory')}</p>
            </div>
          )}
        </div>
      </section>

    </div>
  )
}

export default News